<?php

namespace Svenya\TicketIdGenerator;

class CodeGenerator
{
    protected int $length = 6;
    protected int $count = 1;
    protected array $blacklist = [];
    protected bool $excludeSimilar = true;

    public function __construct(array $config = [])
    {
        $this->length = $config['length'] ?? 6;
        $this->count = $config['count'] ?? 1;
        $this->excludeSimilar = $config['exclude_similar'] ?? true;

        $this->blacklist = $this->loadBlacklist($config['blacklist_categories'] ?? []);
    }

    protected function buildTicketCharset(): array
    {
        $letters = range('A', 'Z');
        $digits = range('2', '9');

        if ($this->excludeSimilar) {
            $letters = array_diff($letters, ['I', 'O', 'L']);
        }

        return array_values(array_merge($letters, $digits));
    }

    protected function buildCodeCharset(): array
    {
        // Mixed case letters and digits (default variant)
        $letters = array_merge(range('A', 'Z'), range('a', 'z'));
        $digits = range('2', '9');

        if ($this->excludeSimilar) {
            $letters = array_diff($letters, ['I', 'O', 'L', 'l']);
        }

        // Re-index the array to ensure consecutive numeric keys
        return array_values(array_merge($letters, $digits));
    }

    // Default charset builder removed; we compute per type on demand

    protected function loadBlacklist(array $selectedCategories = []): array
    {
        $full = include __DIR__ . '/../config/blacklist.php';

        if (empty($selectedCategories)) {
            return array_unique(array_merge(...array_values($full)));
        }

        $blocked = [];
        foreach ($selectedCategories as $category) {
            if (isset($full[$category])) {
                $blocked = array_merge($blocked, $full[$category]);
            }
        }

        return array_unique($blocked);
    }

    public function generate(string $type = null, ?int $length = null): string|array
    {
        if ($this->count === 1) {
            return $this->generateCode($type, $length);
        }

        $codes = [];
        for ($i = 0; $i < $this->count; $i++) {
            $codes[] = $this->generateCode($type, $length);
        }

        return $codes;
    }

    protected function generateCode(string $type = null, ?int $overrideLength = null): string
    {
        $charset = $this->getCharsetForType($type);
        $transform = $this->getTransformForType($type);
        $length = $overrideLength !== null && $overrideLength > 0
            ? $overrideLength
            : $this->getLengthForType($type);

        // Special handling for numeric-only IDs
        if (($type ?? '') === 'numeric') {
            return $this->generateNumericId($length);
        }

        do {
            $code = '';
            for ($i = 0; $i < $length; $i++) {
                $code .= $charset[random_int(0, count($charset) - 1)];
            }
        } while ($this->containsBlacklistedSubstring($code));

        return $transform ? $transform($code) : $code;
    }

    // No legacy type normalization; lean API

    protected function generateNumericId(int $length): string
    {
        // Generate random numeric ID of specified length
        // First digit cannot be 0 to ensure full length
        $code = (string) random_int(1, 9);

        for ($i = 1; $i < $length; $i++) {
            $code .= (string) random_int(0, 9);
        }

        return $code;
    }

    protected function getLengthForType(string $type = null): int
    {
        return ($type === 'numeric') ? 10 : $this->length;
    }

    protected function getCharsetForType(string $type = null): array
    {
        return match ($type) {
            'barcode' => $this->buildTicketCharset(),
            'numeric' => range('0', '9'),
            default => $this->buildCodeCharset(), // default: code
        };
    }

    protected function getTransformForType(string $type = null): ?string
    {
        return $type === 'barcode' ? 'strtoupper' : null;
    }
    protected function containsBlacklistedSubstring(string $code): bool
    {
        $upperCode = strtoupper($code);
        foreach ($this->blacklist as $word) {
            if (str_contains($upperCode, strtoupper($word))) {
                return true;
            }
        }
        return false;
    }
}
