# Svenya ID Generator

[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.0-blue)](https://php.net/)
[![Latest Release](https://img.shields.io/badge/release-v1.2.0-green)](https://github.com/florianlinhardt/svenya-id-generator/releases/tag/v1.2.0)
[![License](https://img.shields.io/badge/license-Proprietary-red)](LICENSE)

A robust PHP library for generating clean, unique, and readable IDs with advanced filtering capabilities. Perfect for ticket systems, order numbers, reference codes, and any application requiring human-friendly identifiers.

## ✨ Key Features

- 🎯 **Multiple ID Types**: Barcode-optimized, mixed-case codes, and numeric IDs
- 🚫 **Smart Blacklist Filtering**: Prevents inappropriate or confusing character combinations
- 🔤 **Similar Character Exclusion**: Automatically excludes I/O/L/l to avoid confusion
- ⚙️ **Highly Configurable**: Customizable length, count, and filtering options
- 🚀 **Laravel Integration**: Native service provider and facade support
- 🧪 **Well Tested**: Comprehensive test suite with Pest PHP

## 📋 Latest Release - v1.2.0 (August 14, 2025)

### 🎉 What's New
- ✅ **Stable Release**: Moved from development builds to production-ready versioning
- 🔧 **Enhanced Testing**: All 21 tests passing with 1,666 assertions
- 📦 **Improved Packaging**: Official GitHub releases with proper semantic versioning
- 🚀 **Production Ready**: Optimized for deployment in production environments

### 🔄 Migration from v1.1.0
No breaking changes! Simply update your `composer.json` to use `^1.2.0` and run `composer update`.

## 📋 Requirements

- PHP 8.0 or higher
- Laravel 8.x or higher (for Laravel integration features)

## 🚀 Installation

Since this is a private package, add the repository to your `composer.json`:

```json
{
  "repositories": [
    {
      "type": "vcs",
      "url": "https://github.com/florianlinhardt/svenya-id-generator"
    }
  ],
  "require": {
    "florianlinhardt/svenya-id-generator": "^1.2.0"
  }
}
```

Then install the package:

```bash
composer install
```

## 🔧 Usage

### Basic Usage

```php
use Svenya\TicketIdGenerator\CodeGenerator;

// Generate a single 6-character ID
$generator = new CodeGenerator();
$id = $generator->generate();
// Returns: "a2B4x9" (mixed case, excludes similar characters)
```

### Configuration Options

```php
$generator = new CodeGenerator([
    'length' => 8,                         // ID length (default: 6)
    'count' => 10,                         // Number of IDs to generate (default: 1)
    'exclude_similar' => true,             // Exclude I/O/L/l characters (default: true)
    'blacklist_categories' => ['vulgar', 'political'] // Filter categories
]);
```

### ID Types

#### 🏷️ Barcode IDs (Recommended for scanning)
```php
$id = $generator->generate('barcode');
// Returns: "A2B4X9QT" (uppercase, no similar chars, optimized for barcode scanners)
```

#### 📝 Mixed Case Codes (Default, human-friendly)
```php
$id = $generator->generate('code');
// Returns: "a2B4x9Qt" (mixed case, good balance of readability and uniqueness)
```

#### 🔢 Numeric IDs (Numbers only)
```php
$id = $generator->generate('numeric');
// Returns: "1847593024" (10-digit numeric ID, no leading zeros)
```

### Batch Generation

```php
$generator = new CodeGenerator(['count' => 5, 'length' => 8]);
$ids = $generator->generate('barcode');
// Returns: ["A2B4X9QT", "P7R5D3MK", "F8N6G4W2", "Z9Y3V7C1", "K5J2H8M6"]
```

### Dynamic Length Override

```php
$generator = new CodeGenerator(['length' => 6]);
$shortId = $generator->generate('code');     // 6 characters
$longId = $generator->generate('code', 12);  // 12 characters (overrides config)
```

## 🔗 Laravel Integration

### Service Provider Registration

The package auto-registers in Laravel. For manual registration, add to `config/app.php`:

```php
'providers' => [
    // ...
    Svenya\TicketIdGenerator\CodeGeneratorServiceProvider::class,
],

'aliases' => [
    // ...
    'CodeGenerator' => Svenya\TicketIdGenerator\Facades\CodeGenerator::class,
],
```

### Using the Facade

```php
use CodeGenerator;

// Generate ticket IDs
$ticketId = CodeGenerator::generate('barcode', 8);

// Generate order numbers
$orderNumber = CodeGenerator::generate('numeric');

// Generate promo codes
$promoCode = CodeGenerator::generate('code', 10);
```

### Dependency Injection

```php
use Svenya\TicketIdGenerator\CodeGenerator;

class TicketService
{
    public function __construct(private CodeGenerator $generator) {}

    public function createTicket(): string
    {
        return $this->generator->generate('barcode', 8);
    }
}
```

## 🚫 Blacklist Filtering

The package includes comprehensive filtering to prevent inappropriate or confusing ID combinations:

### Available Categories

| Category | Description | Examples |
|----------|-------------|----------|
| `political` | Political terms and symbols | KZ, SS, NAZI, AFD |
| `vulgar` | Vulgar language | SEX, FCK, WTF |
| `religious` | Religious terms | GOD, JESUS, ALLAH |
| `drugs` | Drug-related terms | WEED, LSD, MDMA |
| `discrimination` | Discriminatory language | Various offensive terms |
| `internal` | System/internal terms | TEST, TEMP, NULL |

### Usage Example

```php
$generator = new CodeGenerator([
    'length' => 8,
    'blacklist_categories' => ['vulgar', 'political', 'discrimination']
]);

$cleanIds = $generator->generate('barcode');
// All generated IDs will be free of blacklisted substrings
```

## 🎯 Use Cases

### Event Tickets
```php
// Generate scannable ticket IDs
$ticketGenerator = new CodeGenerator([
    'length' => 10,
    'blacklist_categories' => ['vulgar', 'political']
]);
$ticketId = $ticketGenerator->generate('barcode');
```

### Order Numbers
```php
// Generate numeric order IDs
$orderGenerator = new CodeGenerator();
$orderId = $orderGenerator->generate('numeric'); // 10-digit number
```

### Promo Codes
```php
// Generate memorable promo codes
$promoGenerator = new CodeGenerator([
    'length' => 8,
    'exclude_similar' => true
]);
$promoCode = $promoGenerator->generate('code');
```

### Reference Numbers
```php
// Generate multiple reference codes
$refGenerator = new CodeGenerator([
    'length' => 6,
    'count' => 100,
    'blacklist_categories' => ['internal']
]);
$references = $refGenerator->generate('barcode');
```

## 🧪 Testing

Run the comprehensive test suite:

```bash
# Run all tests
composer test

# Run with coverage
./vendor/bin/pest --coverage

# Run specific test file
./vendor/bin/pest tests/CodeGeneratorTest.php
```

## 🔧 Examples

Check the `/examples` directory for practical usage examples:

- `generate_ticket.php` - Basic ticket ID generation
- `generate_event.php` - Event-specific ID generation
- `show_all_ids.php` - Demonstrates all ID types
- `generate_ticket_ids.php` - Batch ticket generation

Run an example:

```bash
cd examples
php show_all_ids.php
```

## 🐛 Troubleshooting

### Common Issues

**IDs contain similar characters**: Ensure `exclude_similar` is set to `true` (default).

**Inappropriate content in IDs**: Add relevant blacklist categories to your configuration.

**Not enough entropy**: Increase the ID length for better uniqueness in large datasets.

**Performance with large batches**: Consider generating IDs in smaller batches for very large quantities (>10,000).

## 🤝 Contributing

This is a private package. For internal contributions:

1. Create a feature branch
2. Add tests for new functionality
3. Run the test suite
4. Submit a pull request

## 📄 License

This package is proprietary software. All rights reserved.

## 🔗 Links

- [Repository](https://github.com/florianlinhardt/svenya-id-generator)
- [Latest Release](https://github.com/florianlinhardt/svenya-id-generator/releases/tag/v1.2.0)
- [All Releases](https://github.com/florianlinhardt/svenya-id-generator/releases)
- [Issues](https://github.com/florianlinhardt/svenya-id-generator/issues)

---

*Built with ❤️ for creating clean, professional ID generation in PHP applications.*
